const canvas = document.querySelector('canvas');
const context = canvas.getContext('2d');

canvas.width = 1024;
canvas.height = 576;

context.fillRect(0, 0, canvas.width, canvas.height);

const gravity = 0.7;

// Check if we have dynamic character/mob data
const useDynamicData = typeof window.ARENA3_DATA !== 'undefined' && window.ARENA3_DATA;
const basePath = useDynamicData ? window.ARENA3_DATA.basePath : '';

// Helper function to build sprite paths
function buildSpritePaths(spriteData, state) {
    if (!useDynamicData || !spriteData || !spriteData.sprites || !spriteData.sprites[state]) {
        console.log(`⚠️ buildSpritePaths: No sprite data for state '${state}'`, {
            useDynamicData,
            hasSpriteData: !!spriteData,
            hasSprites: spriteData ? !!spriteData.sprites : false,
            hasState: spriteData && spriteData.sprites ? !!spriteData.sprites[state] : false
        });
        return null;
    }
    const spriteState = spriteData.sprites[state];
    if (spriteState && spriteState.images) {
        // img paths from getPngSequence start with /, basePath ends with /
        // So we need to remove trailing slash from basePath or handle it
        const cleanBasePath = basePath.replace(/\/$/, '');
        const images = spriteState.images.map(img => cleanBasePath + img);
        console.log(`✅ buildSpritePaths: Found ${images.length} images for state '${state}'`);
        if (state === 'death') {
            console.log(`💀 Death animation images (first 5):`, images.slice(0, 5));
        }
        return images;
    }
    console.log(`⚠️ buildSpritePaths: No images array for state '${state}'`);
    return null;
}

// Helper function to get framesMax
function getFramesMax(spriteData, state) {
    if (!useDynamicData || !spriteData || !spriteData.sprites || !spriteData.sprites[state]) {
        return 1;
    }
    return spriteData.sprites[state].framesMax || 1;
}

// Use dynamic base path if available, otherwise use relative path
const backgroundPath = useDynamicData ? window.ARENA3_DATA.backgroundPath : './backgrounds/Background1.png';
const shopPath = useDynamicData ? window.ARENA3_DATA.shopPath : './backgrounds/shop_anim.png';

console.log('🎨 Background Loading:', {
    useDynamicData,
    backgroundPath,
    shopPath,
    arena3Data: window.ARENA3_DATA
});

const background = new Sprite({
    position: {
        x: 0,
        y: 0
    },
    imageSrc: backgroundPath,
    scale: 1,
    isBackground: true // Flag to identify background
})

const shop = new Sprite({
    position: {
        x: 700,
        y: 170
    },
    imageSrc: shopPath,
    scale: 2.5,
    framesMax: 6
})

// Build player sprites dynamically or use defaults
const playerIdleImages = buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'idle') || [
    './farmer-peasant/idle/01.png',
    './farmer-peasant/idle/02.png',
    './farmer-peasant/idle/03.png',
    './farmer-peasant/idle/04.png',
    './farmer-peasant/idle/05.png'
];

const playerSprites = {
    idle: {
        images: playerIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'idle') || 5
    },
    run: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'run') || playerIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'run') || 5
    },
    jump: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'jump') || [playerIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'jump') || 1
    },
    fall: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'fall') || [playerIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'fall') || 1
    },
    attack1: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'attack1') || [
            './farmer-peasant/fight/01.png',
            './farmer-peasant/fight/02.png',
            './farmer-peasant/fight/03.png',
            './farmer-peasant/fight/04.png',
            './farmer-peasant/fight/05.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'attack1') || 5
    },
    takeHit: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'takeHit') || [
            './farmer-peasant/hurt/01.png',
            './farmer-peasant/hurt/02.png',
            './farmer-peasant/hurt/03.png',
            './farmer-peasant/hurt/04.png',
            './farmer-peasant/hurt/05.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'takeHit') || 5
    },
    death: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'death') || [
            './farmer-peasant/lose/01.png',
            './farmer-peasant/lose/02.png',
            './farmer-peasant/lose/03.png',
            './farmer-peasant/lose/04.png',
            './farmer-peasant/lose/05.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'death') || 5
    },
    win: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'win') || [
            './farmer-peasant/win/01.png',
            './farmer-peasant/win/02.png',
            './farmer-peasant/win/03.png',
            './farmer-peasant/win/04.png',
            './farmer-peasant/win/05.png',
            './farmer-peasant/win/06.png',
            './farmer-peasant/win/07.png',
            './farmer-peasant/win/08.png',
            './farmer-peasant/win/09.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'win') || 9
    }
};

// Get character stats from data or use defaults
const characterStats = useDynamicData && window.ARENA3_DATA.character && window.ARENA3_DATA.character.stats 
    ? window.ARENA3_DATA.character.stats 
    : { hp: 100, attack: 50, defense: 20 };

const player = new Fighter({
    position: {
        x: 100,
        y: 451  // Positioned on the floor (494 - 200 + 157 offset)
    },
    velocity: {
        x: 0,
        y: 0
    },
    offset: {
        x: 0,
        y: 0
    },
    images: playerIdleImages,
    framesMax: playerSprites.idle.framesMax,
    scale: 2.5,
    offset: {
        x: 215,
        y: 157
    },
    sprites: playerSprites,
    health: characterStats.hp,
    attack: characterStats.attack,
    defense: characterStats.defense,
    name: useDynamicData && window.ARENA3_DATA.character ? window.ARENA3_DATA.character.name : 'Player',
    flip: false,  // Player faces right (towards enemy)
    attackBox: {
        offset: {
            x: 100,
            y: 50
        },
        width: 160,
        height: 50
    }
});

// Ensure player starts with idle sprite
console.log('🎮 Initializing player sprite...');
player.switchSprite('idle');
console.log('✅ Player initialized:', {
    name: player.name,
    health: player.health,
    position: player.position,
    hasImage: !!player.image,
    imagesCount: player.images ? player.images.length : 0,
    deathSpriteImages: player.sprites.death.images ? player.sprites.death.images.length : 0,
    deathFramesMax: player.sprites.death.framesMax,
    deathImages: player.sprites.death.images ? player.sprites.death.images.slice(0, 5) : 'none'
});

// Build enemy sprites dynamically or use defaults
const enemyIdleImages = buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'idle') || [
    './desert-rat/idle/left/01.png',
    './desert-rat/idle/left/02.png',
    './desert-rat/idle/left/03.png',
    './desert-rat/idle/left/04.png',
    './desert-rat/idle/left/05.png',
    './desert-rat/idle/left/06.png',
    './desert-rat/idle/left/07.png',
    './desert-rat/idle/left/08.png',
    './desert-rat/idle/left/09.png'
];

const enemySprites = {
    idle: {
        images: enemyIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'idle') || 9
    },
    run: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'run') || enemyIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'run') || 9
    },
    jump: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'jump') || [enemyIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'jump') || 1
    },
    fall: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'fall') || [enemyIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'fall') || 1
    },
    attack1: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'attack1') || [
            './desert-rat/attack/left/01.png',
            './desert-rat/attack/left/02.png',
            './desert-rat/attack/left/03.png',
            './desert-rat/attack/left/04.png',
            './desert-rat/attack/left/05.png',
            './desert-rat/attack/left/06.png',
            './desert-rat/attack/left/07.png',
            './desert-rat/attack/left/08.png',
            './desert-rat/attack/left/09.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'attack1') || 9
    },
    takeHit: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'takeHit') || [
            './desert-rat/idle/left/01.png',
            './desert-rat/idle/left/02.png',
            './desert-rat/idle/left/03.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'takeHit') || 3
    },
    death: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'death') || [
            './desert-rat/idle/left/01.png',
            './desert-rat/idle/left/02.png',
            './desert-rat/idle/left/03.png',
            './desert-rat/idle/left/04.png',
            './desert-rat/idle/left/05.png',
            './desert-rat/idle/left/06.png',
            './desert-rat/idle/left/07.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'death') || 7
    }
};

// Get mob stats from data or use defaults
const mobStats = useDynamicData && window.ARENA3_DATA.mob && window.ARENA3_DATA.mob.stats 
    ? window.ARENA3_DATA.mob.stats 
    : { hp: 100, attack: 40, defense: 20 };

const enemy = new Fighter({
    position: {
        x: 800,
        y: 464  // Positioned on the floor (494 - 200 + 170 offset)
    },
    velocity: {
        x: 0,
        y: 0
    },
    color: 'blue',
    offset: {
        x: 0,
        y: 0
    },
    images: enemyIdleImages,
    framesMax: enemySprites.idle.framesMax,
    scale: 2.5, // Same scale as player (will be 200px)
    offset: {
        x: 215,
        y: 170
    },
    sprites: enemySprites,
    health: mobStats.hp,
    attack: mobStats.attack,
    defense: mobStats.defense,
    name: useDynamicData && window.ARENA3_DATA.mob ? window.ARENA3_DATA.mob.name : 'Enemy',
    flip: true,  // Flip enemy horizontally to face left (towards player)
    attackBox: {
        offset: {
            x: -170,
            y: 50
        },
        width: 170,
        height: 50
    }
})

// Ensure enemy starts with idle sprite and is visible immediately
console.log('🎮 Initializing enemy sprite...');
enemy.switchSprite('idle');
console.log('✅ Enemy initialized:', {
    name: enemy.name,
    health: enemy.health,
    position: enemy.position,
    hasImage: !!enemy.image,
    imagesCount: enemy.images ? enemy.images.length : 0,
    deathSpriteImages: enemy.sprites.death.images ? enemy.sprites.death.images.length : 0,
    deathFramesMax: enemy.sprites.death.framesMax
});

// Force enemy to be visible by ensuring images are set
if (enemy.images && enemy.images.length > 0) {
    // Images are already set - check if they're loaded
    const firstImage = enemy.images[0];
    console.log('✅ Enemy images loaded:', enemy.images.length, '| First image complete:', firstImage ? firstImage.complete : false, '| naturalWidth:', firstImage ? firstImage.naturalWidth : 0);
    
    // If images aren't loaded yet, wait for them and force redraw when loaded
    if (firstImage && !firstImage.complete) {
        firstImage.onload = function() {
            console.log('✅ Enemy first image now loaded! naturalWidth:', this.naturalWidth);
            // Force redraw by switching sprite
            enemy.switchSprite('idle');
        };
    }
} else if (enemy.image) {
    // Single image sprite sheet
    console.log('✅ Enemy image loaded (sprite sheet) | Complete:', enemy.image.complete);
    if (!enemy.image.complete) {
        enemy.image.onload = function() {
            console.log('✅ Enemy sprite sheet now loaded!');
        };
    }
} else {
    console.warn('⚠️ Enemy has no images! Attempting to load...');
    // Force load idle images
    if (enemy.sprites && enemy.sprites.idle) {
        if (enemy.sprites.idle.images && enemy.sprites.idle.images.length > 0) {
            enemy.images = enemy.sprites.idle.images;
            enemy.image = enemy.sprites.idle.images[0];
            enemy.framesMax = enemy.sprites.idle.framesMax;
            console.log('✅ Enemy images forced loaded from sprites');
        }
    }
}

// Log initialization
console.log('🎮 Arena 3 Game Initialized');
console.log('👤 Player:', {
    name: useDynamicData ? window.ARENA3_DATA.character.name : 'Default',
    health: player.health,
    attack: player.attackPower,
    defense: player.defense,
    position: player.position
});
console.log('👹 Enemy:', {
    name: useDynamicData ? window.ARENA3_DATA.mob.name : 'Default',
    health: enemy.health,
    attack: enemy.attackPower,
    defense: enemy.defense,
    position: enemy.position,
    sprite: enemy.image ? 'loaded' : 'not loaded',
    images: enemy.images ? enemy.images.length + ' images' : 'no images'
});

// Automatic fighting system - no keyboard controls needed
let currentRound = 1;
let isPlayerTurn = true;
let roundInProgress = false;
let roundCooldown = 0;
let attackCooldown = 0;
let attackExecuted = false;
let returningToPosition = false;
const ROUND_DELAY = 90; // frames between rounds (~1.5 seconds at 60fps) - faster
const ATTACK_DELAY = 15; // frames before attack executes - faster
const ATTACK_DISTANCE = 300; // Distance to maintain for attacks (increased to get closer)
const ATTACK_LUNGE_DISTANCE = 80; // How far forward to move when attacking

// Finish Move System
let playerFinishMeter = 0; // 0-100
let enemyFinishMeter = 0; // 0-100
let playerFinishMoveReady = false;
let enemyFinishMoveReady = false;
let finishMoveActive = false; // Track if finish move is currently executing
const FINISH_MOVE_CHARGE_PER_ATTACK = 30; // Fill 30% per attack (faster charging)
const FINISH_MOVE_DAMAGE_MULTIPLIER = 2.5; // Finish move does 2.5x damage

// Game over state
let gameOver = false;
let winner = null; // 'player' or 'enemy'
let winnerReturningToPosition = false; // Track if winner is returning to starting position
let winnerReachedPosition = false; // Track if winner has reached starting position

// Store starting positions
const playerStartX = 100;
const enemyStartX = 800;
// Return positions (closer to center, not all the way back)
const playerReturnX = 250; // Player returns to this position (not all the way left)
const enemyReturnX = 750; // Enemy returns to this position (not all the way right)

// Timer removed - game ends when health reaches 0
// decreaseTimer();

// Track current state for header display (declared outside animate function)
let playerCurrentState = 'idle';
let enemyCurrentState = 'idle';

// Initialize health bars
setTimeout(() => {
    if (player && player.maxHealth) {
        const playerHealthPercent = (player.health / player.maxHealth) * 100;
        gsap.set('#playerHealth', { width: playerHealthPercent + '%' });
    }
    if (enemy && enemy.maxHealth) {
        const enemyHealthPercent = (enemy.health / enemy.maxHealth) * 100;
        gsap.set('#enemyHealth', { width: enemyHealthPercent + '%' });
    }
}, 100);

function handleAutomaticFighting() {
    // Don't fight if game is over - winner stays in place with victory animation
    if (gameOver) {
        // Ensure winner keeps playing win animation and stays in place
        if (winner === 'player' && playerCurrentState !== 'win') {
            player.switchSprite('win');
            playerCurrentState = 'win';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('win');
            }
        }
        return;
    }
    
    // Don't fight if someone is dead
    if (player.dead || enemy.dead) {
        return;
    }

    // Handle round cooldown
    if (roundCooldown > 0) {
        roundCooldown--;
        return;
    }

    const attacker = isPlayerTurn ? player : enemy;
    const defender = isPlayerTurn ? enemy : player;
    const attackerStartX = isPlayerTurn ? playerStartX : enemyStartX;
    const attackerReturnX = isPlayerTurn ? playerReturnX : enemyReturnX;
    const defenderStartX = isPlayerTurn ? enemyStartX : playerStartX;
    
    // Calculate distance between fighters
    const distance = Math.abs(attacker.position.x - defender.position.x);
    
    // Log round start
    if (!roundInProgress && distance <= ATTACK_DISTANCE) {
        console.log(`⚔️ Round ${currentRound} - ${isPlayerTurn ? 'Player' : 'Enemy'} turn`);
    }
    
    // If returning to position after attack
    if (returningToPosition) {
        const distanceToReturn = Math.abs(attacker.position.x - attackerReturnX);
        if (distanceToReturn > 5) {
            // Move back to return position (not all the way to starting position)
            if (attacker.position.x < attackerReturnX) {
                attacker.velocity.x = 8; // Faster return
            } else if (attacker.position.x > attackerReturnX) {
                attacker.velocity.x = -8; // Faster return
            }
        } else {
            // Reached return position
            attacker.position.x = attackerReturnX;
            attacker.velocity.x = 0;
            returningToPosition = false;
            isPlayerTurn = !isPlayerTurn;
            roundInProgress = false;
            roundCooldown = ROUND_DELAY;
            currentRound++;
            attackExecuted = false;
            // Update round display
            if (typeof window.updateRoundDisplay === 'function') {
                window.updateRoundDisplay(currentRound);
            }
            console.log(`🔄 Turn switched - Now ${isPlayerTurn ? 'Player' : 'Enemy'} turn | Round ${currentRound}`);
            
            // Add round change to fight log
            if (typeof window.addFightLogEntry === 'function') {
                window.addFightLogEntry(`--- Round ${currentRound} ---`, 'normal');
            }
        }
        defender.velocity.x = 0;
        return;
    }
    
    // If too far, move closer to center
    if (distance > ATTACK_DISTANCE) {
        roundInProgress = false;
        attackExecuted = false;
        attackCooldown = 0;
        
        if (isPlayerTurn) {
            // Player moves right toward center
            if (player.position.x < enemy.position.x - ATTACK_DISTANCE) {
                player.velocity.x = 8; // Faster movement
            } else {
                player.velocity.x = 0;
            }
            enemy.velocity.x = 0;
        } else {
            // Enemy moves left toward center
            if (enemy.position.x > player.position.x + ATTACK_DISTANCE) {
                enemy.velocity.x = -8; // Faster movement
            } else {
                enemy.velocity.x = 0;
            }
            player.velocity.x = 0;
        }
    } else {
        // Close enough, perform attack
        if (!roundInProgress) {
            roundInProgress = true;
            attackCooldown = ATTACK_DELAY;
            attackExecuted = false;
        }
        
        // Calculate minimum distance to maintain (don't pass through opponent)
        // Maintain comfortable distance for both player and enemy
        const minDistance = isPlayerTurn ? 100 : 80; // Player maintains 100px distance, enemy 80px
        const currentDistance = Math.abs(attacker.position.x - defender.position.x);
        
        // Move forward during attack
        if (attackCooldown > 0) {
            attackCooldown--;
            // Start moving forward, but stop if too close
            if (currentDistance > minDistance) {
                if (isPlayerTurn) {
                    attacker.velocity.x = 8; // Moderate attack movement for player
                } else {
                    attacker.velocity.x = -10; // Moderate attack movement for enemy
                }
            } else {
                attacker.velocity.x = 0; // Stop if too close
            }
            defender.velocity.x = 0;
        } else if (!attackExecuted) {
            // Execute attack
            if (!attacker.dead && !defender.dead && !attacker.isAttacking) {
                // Check if finish move is ready - use it automatically
                const finishMoveReady = isPlayerTurn ? playerFinishMoveReady : enemyFinishMoveReady;
                const attackerName = attacker.name || (isPlayerTurn ? 'Player' : 'Enemy');
                
                if (finishMoveReady && !finishMoveActive) {
                    console.log(`🔥🔥🔥 FINISH MOVE ACTIVATED! ${isPlayerTurn ? 'Player' : 'Enemy'} unleashes special attack! 🔥🔥🔥`);
                    if (typeof window.addFightLogEntry === 'function') {
                        window.addFightLogEntry(`⚡ ${attackerName} prepares FINISH MOVE!`, 'attack');
                    }
                } else {
                    console.log(`🗡️ ${isPlayerTurn ? 'Player' : 'Enemy'} attacks!`);
                    if (typeof window.addFightLogEntry === 'function') {
                        window.addFightLogEntry(`${attackerName} attacks!`, 'attack');
                    }
                }
                attacker.attack();
                attackExecuted = true;
                // Continue moving forward during attack, but maintain minimum distance
                if (currentDistance > minDistance) {
                    if (isPlayerTurn) {
                        attacker.velocity.x = 8; // Moderate speed for player
                    } else {
                        attacker.velocity.x = -8; // Moderate speed for enemy
                    }
                } else {
                    attacker.velocity.x = 0; // Stop if too close
                }
            }
        } else {
            // Keep moving forward for a bit during attack animation to ensure hit connects
            const attackFramesMax = attacker.sprites.attack1.framesMax;
            if (attacker.frameCurrent < attackFramesMax - 1) {
                // Still in attack animation, keep moving forward but maintain minimum distance
                if (currentDistance > minDistance) {
                    if (isPlayerTurn) {
                        attacker.velocity.x = 6; // Slower movement during attack for player
                    } else {
                        attacker.velocity.x = -6; // Slower movement for enemy
                    }
                } else {
                    // Stop if at minimum distance
                    attacker.velocity.x = 0;
                }
            } else {
                // Attack animation done, slow down
                attacker.velocity.x *= 0.9;
            }
        }
        
        // Wait for attack animation to complete, then return to position
        const attackFramesMax = attacker.sprites.attack1.framesMax;
        const attackComplete = attackExecuted && 
            (attacker.image !== attacker.sprites.attack1.image || 
             attacker.frameCurrent >= attackFramesMax - 1);
        
        // Also wait for defender's hit animation to complete if they were hit
        const defenderHitFramesMax = defender.sprites.takeHit.framesMax;
        const hitAnimationComplete = defender.image !== defender.sprites.takeHit.image ||
            defender.frameCurrent >= defenderHitFramesMax - 1;
        
        if (attackComplete && !attacker.isAttacking && hitAnimationComplete) {
            // Start returning to starting position
            console.log(`✅ Round ${currentRound} complete - ${isPlayerTurn ? 'Player' : 'Enemy'} finished attack`);
            returningToPosition = true;
            attacker.velocity.x = 0;
        }
    }
}

let frameCount = 0;
function animate() {
    window.requestAnimationFrame(animate);
    frameCount++;
    
    context.fillStyle = 'black';
    context.fillRect(0, 0, canvas.width, canvas.height);
    background.update();
    shop.update();
    
    // Handle automatic fighting (this sets velocities)
    if (!gameOver) {
        handleAutomaticFighting();
    } else {
        // Game over - make winner go to center and perform jumping victory moves
        const winnerFighter = winner === 'player' ? player : enemy;
        const centerX = canvas.width / 2; // Center of canvas
        const loserFighter = winner === 'player' ? enemy : player;
        
        // Stop loser movement completely
        loserFighter.velocity.x = 0;
        loserFighter.velocity.y = 0;
        
        // If winner hasn't started moving to center, initiate movement
        if (!winnerReturningToPosition && !winnerReachedPosition) {
            winnerReturningToPosition = true;
            console.log(`🏆 Winner (${winner}) starting to move to center position ${centerX}`);
        }
        
        // If winner is moving to center, move them there
        if (winnerReturningToPosition && !winnerReachedPosition) {
            const distanceToCenter = Math.abs(winnerFighter.position.x - centerX);
            
            if (distanceToCenter > 10) {
                // Move towards center
                if (winnerFighter.position.x < centerX) {
                    winnerFighter.velocity.x = 8; // Move right
                } else if (winnerFighter.position.x > centerX) {
                    winnerFighter.velocity.x = -8; // Move left
                }
                // Play run animation while moving to center
                if (winner === 'player' && playerCurrentState !== 'run') {
                    player.switchSprite('run');
                    playerCurrentState = 'run';
                    if (typeof window.updateCharacterState === 'function') {
                        window.updateCharacterState('run');
                    }
                } else if (winner === 'enemy' && enemyCurrentState !== 'run') {
                    enemy.switchSprite('run');
                    enemyCurrentState = 'run';
                    if (typeof window.updateMobState === 'function') {
                        window.updateMobState('run');
                    }
                }
            } else {
                // Reached center - position at center and start victory jumping
                winnerFighter.position.x = centerX;
                winnerFighter.velocity.x = 0;
                winnerReturningToPosition = false;
                winnerReachedPosition = true;
                
                // Switch to win animation
                if (winnerFighter.sprites && winnerFighter.sprites.win) {
                    winnerFighter.switchSprite('win');
                    if (winner === 'player') {
                        playerCurrentState = 'win';
                        if (typeof window.updateCharacterState === 'function') {
                            window.updateCharacterState('win');
                        }
                    } else {
                        enemyCurrentState = 'win';
                        if (typeof window.updateMobState === 'function') {
                            window.updateMobState('win');
                        }
                    }
                    console.log(`🎉 Winner (${winner}) reached center and starting victory jumps!`);
                } else {
                    // Fallback to idle if no win animation
                    winnerFighter.switchSprite('idle');
                    if (winner === 'player') {
                        playerCurrentState = 'idle';
                        if (typeof window.updateCharacterState === 'function') {
                            window.updateCharacterState('idle');
                        }
                    } else {
                        enemyCurrentState = 'idle';
                        if (typeof window.updateMobState === 'function') {
                            window.updateMobState('idle');
                        }
                    }
                }
            }
        } else if (winnerReachedPosition) {
            // Winner has reached center - keep them at center and play victory animation
            winnerFighter.position.x = centerX; // Keep at center
            winnerFighter.velocity.x = 0; // No horizontal movement
            winnerFighter.velocity.y = 0; // No vertical movement (stay on ground)
            
            // Keep winner on the ground at proper Y position
            const floorY = canvas.height - 82; // 494
            const characterHeight = 200;
            const targetY = floorY - characterHeight + (winnerFighter.offset ? winnerFighter.offset.y : 0);
            winnerFighter.position.y = targetY; // Keep on ground
            
            // Ensure win animation is playing continuously
            if (winner === 'player') {
                if (player.sprites && player.sprites.win && playerCurrentState !== 'win') {
                    player.switchSprite('win');
                    playerCurrentState = 'win';
                    if (typeof window.updateCharacterState === 'function') {
                        window.updateCharacterState('win');
                    }
                } else if (playerCurrentState !== 'win' && player.sprites && player.sprites.win) {
                    // Force win animation if not already playing
                    player.switchSprite('win');
                    playerCurrentState = 'win';
                    if (typeof window.updateCharacterState === 'function') {
                        window.updateCharacterState('win');
                    }
                }
            } else if (winner === 'enemy') {
                if (enemy.sprites && enemy.sprites.win && enemyCurrentState !== 'win') {
                    enemy.switchSprite('win');
                    enemyCurrentState = 'win';
                    if (typeof window.updateMobState === 'function') {
                        window.updateMobState('win');
                    }
                } else if (enemyCurrentState !== 'win' && enemy.sprites && enemy.sprites.win) {
                    // Force win animation if not already playing
                    enemy.switchSprite('win');
                    enemyCurrentState = 'win';
                    if (typeof window.updateMobState === 'function') {
                        window.updateMobState('win');
                    }
                }
            }
        }
    }
    
    // Log enemy visibility on first few frames
    if (frameCount <= 10) {
        const currentImg = enemy.images && enemy.images[0] ? enemy.images[0] : null;
        console.log(`Frame ${frameCount}: Enemy draw check -`, {
            hasImages: !!enemy.images,
            imagesLength: enemy.images ? enemy.images.length : 0,
            hasImage: !!enemy.image,
            currentImage: currentImg ? 'exists' : 'missing',
            imageComplete: currentImg ? currentImg.complete : false,
            imageNaturalWidth: currentImg ? currentImg.naturalWidth : 0,
            imageNaturalHeight: currentImg ? currentImg.naturalHeight : 0,
            position: enemy.position,
            offset: enemy.offset,
            drawX: enemy.position.x - enemy.offset.x,
            drawY: enemy.position.y - enemy.offset.y
        });
    }
    
    player.update();
    enemy.update();

    // Handle sprite animations based on velocity and state
    // Don't override win animation if player won and reached center
    if (gameOver && winner === 'player' && winnerReachedPosition) {
        // Keep win animation playing - don't change sprite based on velocity
        if (playerCurrentState !== 'win' && player.sprites && player.sprites.win) {
            player.switchSprite('win');
            playerCurrentState = 'win';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('win');
            }
        }
    } else if (gameOver && winner === 'player' && winnerReturningToPosition) {
        // Player is returning to center - let the animate() function handle the animation
        // Don't override here
    } else if (player.isAttacking) {
        // Don't change sprite during attack
        if (playerCurrentState !== 'attack1') {
            playerCurrentState = 'attack1';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('attack1');
            }
        }
    } else if (player.dead && !(gameOver && winner === 'player')) {
        if (playerCurrentState !== 'death') {
            console.log('💀 Player is dead, setting state to death');
            playerCurrentState = 'death';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('death');
            }
        }
    } else if (player.velocity.x !== 0) {
        player.switchSprite('run');
        if (playerCurrentState !== 'run') {
            playerCurrentState = 'run';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('run');
            }
        }
    } else if (player.velocity.y < 0) {
        player.switchSprite('jump');
        if (playerCurrentState !== 'jump') {
            playerCurrentState = 'jump';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('jump');
            }
        }
    } else if (player.velocity.y > 0) {
        player.switchSprite('fall');
        if (playerCurrentState !== 'fall') {
            playerCurrentState = 'fall';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('fall');
            }
        }
    } else if (!player.dead && !gameOver && playerCurrentState !== 'win') {
        player.switchSprite('idle');
        if (playerCurrentState !== 'idle') {
            playerCurrentState = 'idle';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('idle');
            }
        }
    }

    if (enemy.isAttacking) {
        // Don't change sprite during attack
        if (enemyCurrentState !== 'attack1') {
            enemyCurrentState = 'attack1';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('attack1');
            }
        }
    } else if (gameOver && winner === 'enemy' && winnerReachedPosition) {
        // Keep win animation playing - don't change sprite based on velocity
        if (enemyCurrentState !== 'win' && enemy.sprites && enemy.sprites.win) {
            enemy.switchSprite('win');
            enemyCurrentState = 'win';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('win');
            }
        }
    } else if (gameOver && winner === 'enemy' && winnerReturningToPosition) {
        // Enemy is returning to center - let the animate() function handle the animation
        // Don't override here
    } else if (enemy.dead && !(gameOver && winner === 'enemy')) {
        if (enemyCurrentState !== 'death') {
            console.log('💀 Enemy is dead, setting state to death');
            enemyCurrentState = 'death';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('death');
            }
        }
    } else if (enemy.velocity.x !== 0) {
        enemy.switchSprite('run');
        if (enemyCurrentState !== 'run') {
            enemyCurrentState = 'run';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('run');
            }
        }
    } else if (enemy.velocity.y < 0) {
        enemy.switchSprite('jump');
        if (enemyCurrentState !== 'jump') {
            enemyCurrentState = 'jump';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('jump');
            }
        }
    } else if (enemy.velocity.y > 0) {
        enemy.switchSprite('fall');
        if (enemyCurrentState !== 'fall') {
            enemyCurrentState = 'fall';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('fall');
            }
        }
    } else if (!enemy.dead && !gameOver) {
        enemy.switchSprite('idle');
        if (enemyCurrentState !== 'idle') {
            enemyCurrentState = 'idle';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('idle');
            }
        }
    }

    //detect colission & enemy hit
    // Check frames 1, 2, 3, or 4 for player attack (5 frame animation)
    // Check collision more frequently during attack
    if (player.isAttacking && (player.frameCurrent >= 1 && player.frameCurrent <= 4)) {
        if (rectangularCollision({
            rectangle1: player,
            rectangle2: enemy
        })) {
            // Calculate damage using Arena 1 system
            const attackerPower = player.attackPower || 50;
            const attackerLevel = 1; // Default level
            let damage = Math.floor(attackerPower * (0.8 + Math.random() * 0.4));
            
            // Level boost
            const levelBoost = 1 + (attackerLevel * 0.05);
            damage = Math.floor(damage * levelBoost);
            
            // Critical hit chance (20%)
            let isCritical = false;
            if (Math.random() < 0.2) {
                damage = Math.floor(damage * 1.75);
                isCritical = true;
            }
            
            // Minimum damage
            damage = Math.max(damage, 5);
            
            // Check if this is a finish move
            const isFinishMove = playerFinishMoveReady && !finishMoveActive;
            if (isFinishMove) {
                damage = Math.floor(damage * FINISH_MOVE_DAMAGE_MULTIPLIER);
                console.log(`🔥🔥🔥 FINISH MOVE! Player deals ${damage} damage! 🔥🔥🔥`);
                playerFinishMeter = 0;
                playerFinishMoveReady = false;
                finishMoveActive = true;
                // Update meter display
                if (typeof window.updatePlayerFinishMoveMeter === 'function') {
                    window.updatePlayerFinishMoveMeter(0);
                }
                // Reset finish move flag after animation
                setTimeout(() => {
                    finishMoveActive = false;
                }, 2000);
            } else {
                // Regular attack - charge finish move meter
                playerFinishMeter = Math.min(100, playerFinishMeter + FINISH_MOVE_CHARGE_PER_ATTACK);
                if (playerFinishMeter >= 100 && !playerFinishMoveReady) {
                    playerFinishMoveReady = true;
                    console.log('⚡ FINISH MOVE READY! Special attack will trigger automatically!');
                    if (typeof window.addFightLogEntry === 'function') {
                        const playerName = player.name || 'Player';
                        window.addFightLogEntry(`⚡ ${playerName}'s FINISH MOVE is ready!`, 'finish-move');
                    }
                }
                // Update meter display
                if (typeof window.updatePlayerFinishMoveMeter === 'function') {
                    window.updatePlayerFinishMoveMeter(playerFinishMeter);
                }
            }
            
            console.log(`💥 Player hits Enemy for ${damage} damage${isCritical ? ' (CRITICAL!)' : ''}${isFinishMove ? ' (FINISH MOVE!)' : ''} | Enemy HP: ${enemy.health} -> ${enemy.health - damage}`);
            
            // Add to fight log
            if (typeof window.addFightLogEntry === 'function') {
                const playerName = player.name || 'Player';
                const enemyName = enemy.name || 'Enemy';
                let logMessage = `${playerName} hits ${enemyName} for ${damage} damage`;
                let logType = 'hit';
                
                if (isFinishMove) {
                    logMessage = `🔥 ${playerName} uses FINISH MOVE on ${enemyName} for ${damage} damage!`;
                    logType = 'finish-move';
                } else if (isCritical) {
                    logMessage = `⚡ ${playerName} CRITICAL HIT on ${enemyName} for ${damage} damage!`;
                    logType = 'critical';
                }
                
                window.addFightLogEntry(logMessage, logType);
            }
            
            // Create slash effect
            if (typeof window.createSlashEffect === 'function') {
                const attackerX = player.position.x;
                const targetX = enemy.position.x;
                const targetY = enemy.position.y + enemy.height / 2;
                window.createSlashEffect(attackerX, targetX, targetY, isCritical || isFinishMove);
            }
            
            enemy.takeHit(damage);
            player.isAttacking = false;
            // Update state to show hit
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('takeHit');
                setTimeout(() => {
                    if (enemyCurrentState !== 'death') {
                        window.updateMobState(enemyCurrentState);
                    }
                }, 300);
            }
            // Update health bar based on percentage
            const healthPercent = (enemy.health / enemy.maxHealth) * 100;
            gsap.to('#enemyHealth', {
                width: healthPercent + '%'
            })
        }
    }

    // if player misses - check at end of attack animation
    if (player.isAttacking && player.frameCurrent >= 4) {
        player.isAttacking = false;
    }

    // enemy hits
    // Check frames 2, 3, 4, 5, 6, 7, or 8 for enemy attack (9 frame animation)
    // Check collision more frequently during attack
    if (enemy.isAttacking && (enemy.frameCurrent >= 2 && enemy.frameCurrent <= 8)) {
        if (rectangularCollision({
            rectangle1: enemy,
            rectangle2: player
        })) {
            // Calculate damage using Arena 1 system
            const attackerPower = enemy.attackPower || 40;
            const attackerLevel = 1; // Default level
            let damage = Math.floor(attackerPower * (0.8 + Math.random() * 0.4));
            
            // Level boost
            const levelBoost = 1 + (attackerLevel * 0.05);
            damage = Math.floor(damage * levelBoost);
            
            // Critical hit chance (20%)
            let isCritical = false;
            if (Math.random() < 0.2) {
                damage = Math.floor(damage * 1.75);
                isCritical = true;
            }
            
            // Minimum damage
            damage = Math.max(damage, 5);
            
            // Check if this is a finish move
            const isFinishMove = enemyFinishMoveReady && !finishMoveActive;
            if (isFinishMove) {
                damage = Math.floor(damage * FINISH_MOVE_DAMAGE_MULTIPLIER);
                console.log(`🔥🔥🔥 FINISH MOVE! Enemy deals ${damage} damage! 🔥🔥🔥`);
                enemyFinishMeter = 0;
                enemyFinishMoveReady = false;
                finishMoveActive = true;
                // Update meter display
                if (typeof window.updateEnemyFinishMoveMeter === 'function') {
                    window.updateEnemyFinishMoveMeter(0);
                }
                // Reset finish move flag after animation
                setTimeout(() => {
                    finishMoveActive = false;
                }, 2000);
            } else {
                // Regular attack - charge finish move meter
                enemyFinishMeter = Math.min(100, enemyFinishMeter + FINISH_MOVE_CHARGE_PER_ATTACK);
                if (enemyFinishMeter >= 100 && !enemyFinishMoveReady) {
                    enemyFinishMoveReady = true;
                    console.log('⚡ ENEMY FINISH MOVE READY!');
                    if (typeof window.addFightLogEntry === 'function') {
                        const enemyName = enemy.name || 'Enemy';
                        window.addFightLogEntry(`⚡ ${enemyName}'s FINISH MOVE is ready!`, 'finish-move');
                    }
                }
                // Update meter display
                if (typeof window.updateEnemyFinishMoveMeter === 'function') {
                    window.updateEnemyFinishMoveMeter(enemyFinishMeter);
                }
            }
            
            console.log(`💥 Enemy hits Player for ${damage} damage${isCritical ? ' (CRITICAL!)' : ''}${isFinishMove ? ' (FINISH MOVE!)' : ''} | Player HP: ${player.health} -> ${player.health - damage}`);
            
            // Add to fight log
            if (typeof window.addFightLogEntry === 'function') {
                const playerName = player.name || 'Player';
                const enemyName = enemy.name || 'Enemy';
                let logMessage = `${enemyName} hits ${playerName} for ${damage} damage`;
                let logType = 'hit';
                
                if (isFinishMove) {
                    logMessage = `🔥 ${enemyName} uses FINISH MOVE on ${playerName} for ${damage} damage!`;
                    logType = 'finish-move';
                } else if (isCritical) {
                    logMessage = `⚡ ${enemyName} CRITICAL HIT on ${playerName} for ${damage} damage!`;
                    logType = 'critical';
                }
                
                window.addFightLogEntry(logMessage, logType);
            }
            
            // Create slash effect
            if (typeof window.createSlashEffect === 'function') {
                const attackerX = enemy.position.x;
                const targetX = player.position.x;
                const targetY = player.position.y + player.height / 2;
                window.createSlashEffect(attackerX, targetX, targetY, isCritical || isFinishMove);
            }
            
            player.takeHit(damage);
            enemy.isAttacking = false;
            // Update state to show hit
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('takeHit');
                setTimeout(() => {
                    if (playerCurrentState !== 'death') {
                        window.updateCharacterState(playerCurrentState);
                    }
                }, 300);
            }
            // Update health bar based on percentage
            const healthPercent = (player.health / player.maxHealth) * 100;
            gsap.to('#playerHealth', {
                width: healthPercent + '%'
            })
        }
    }

    // if enemy misses - check at end of attack animation
    if (enemy.isAttacking && enemy.frameCurrent >= 8) {
        enemy.isAttacking = false;
    }

    // end game based on health
    if ((enemy.health <= 0 || player.health <= 0) && !gameOver) {
        console.log('🎮 GAME OVER CHECK:', {
            playerHealth: player.health,
            enemyHealth: enemy.health,
            playerDead: player.dead,
            enemyDead: enemy.dead,
            gameOver: gameOver
        });
        
        gameOver = true;
        winner = player.health > enemy.health ? 'player' : 'enemy';
        const winnerFighter = winner === 'player' ? player : enemy;
        const loserFighter = winner === 'player' ? enemy : player;
        const winnerName = winnerFighter.name || (winner === 'player' ? 'Player' : 'Enemy');
        console.log(`🏆 Game Over! ${winnerName} wins! | Player HP: ${player.health} | Enemy HP: ${enemy.health}`);
        
        // Format victory message - if player wins, show "player (name) win"
        let victoryMessage;
        let fightLogMessage;
        if (winner === 'player') {
            victoryMessage = `<span style="color: #60a5fa; font-weight: bold; font-size: 24px;">player (${winnerName}) win 🏆</span>`;
            fightLogMessage = `🏆 player (${winnerName}) win`;
        } else {
            victoryMessage = `<span style="color: #f87171; font-weight: bold; font-size: 24px;">${winnerName} Wins! 🏆</span>`;
            fightLogMessage = `🏆 ${winnerName} wins the fight!`;
        }
        
        // Update game status with winner name
        if (typeof window.updateGameStatus === 'function') {
            const gameStatusElement = document.getElementById('gameStatus');
            if (gameStatusElement) {
                gameStatusElement.innerHTML = victoryMessage;
            }
        }
        
        // Add victory entry to fight log
        if (typeof window.addFightLogEntry === 'function') {
            window.addFightLogEntry(fightLogMessage, 'victory');
        }
        
        // Award resources for victory (only if player wins)
        if (winner === 'player') {
            fetch('/characters/arena-victory', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                    'Accept': 'application/json',
                },
                credentials: 'same-origin',
            })
            .then(response => {
                if (response.ok) {
                    return response.json();
                } else {
                    throw new Error('Failed to award victory rewards');
                }
            })
            .then(data => {
                // Add reward messages to fight log
                if (typeof window.addFightLogEntry === 'function') {
                    window.addFightLogEntry(`⚔️ Attack +${data.rewards.attack}!`, 'victory');
                    window.addFightLogEntry(`🛡️ Defense +${data.rewards.defense}!`, 'victory');
                    window.addFightLogEntry(`💪 Stamina +${data.rewards.stamina}!`, 'victory');
                    window.addFightLogEntry(`✨ Resources increased!`, 'victory');
                }
                console.log('✅ Victory rewards awarded:', data.rewards);
            })
            .catch(error => {
                console.error('Error awarding victory rewards:', error);
            });
        }
        
        // Prevent winner from being marked as dead
        
        console.log('👤 Winner Fighter State:', {
            name: winnerFighter.name,
            health: winnerFighter.health,
            dead: winnerFighter.dead,
            position: winnerFighter.position,
            currentSprite: winnerFighter.image ? 'has image' : 'no image',
            images: winnerFighter.images ? winnerFighter.images.length + ' images' : 'no images'
        });
        
        // Ensure winner is NOT dead and has at least 1 HP
        const wasDead = winnerFighter.dead;
        winnerFighter.dead = false;
        if (winnerFighter.health <= 0) {
            console.log(`⚠️ Winner health was ${winnerFighter.health}, setting to 1`);
            winnerFighter.health = 1; // Give winner at least 1 HP so they don't disappear
        }
        console.log(`✅ Winner dead state: ${wasDead} -> ${winnerFighter.dead}, health: ${winnerFighter.health}`);
        
        // Stop all movement
        player.velocity.x = 0;
        enemy.velocity.x = 0;
        player.velocity.y = 0;
        enemy.velocity.y = 0;
        
        // Make sure winner is visible and not in death state
        console.log('🔄 Switching winner sprite...');
        const beforeSprite = winnerFighter.image ? 'has image' : 'no image';
        
        // If player wins, use win animation; otherwise use idle
        if (winner === 'player' && winnerFighter.sprites && winnerFighter.sprites.win) {
            console.log('🏆 Player wins! Switching to win animation...');
            winnerFighter.switchSprite('win');
            playerCurrentState = 'win';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('win');
            }
        } else {
            winnerFighter.switchSprite('idle');
            if (winner === 'player') {
                playerCurrentState = 'idle';
                console.log('✅ Player state set to idle');
                if (typeof window.updateCharacterState === 'function') {
                    window.updateCharacterState('idle');
                }
            } else {
                enemyCurrentState = 'idle';
                console.log('✅ Enemy state set to idle');
                if (typeof window.updateMobState === 'function') {
                    window.updateMobState('idle');
                }
            }
        }
        
        const afterSprite = winnerFighter.image ? 'has image' : 'no image';
        console.log(`✅ Winner sprite switch: ${beforeSprite} -> ${afterSprite}`);
        
        // Initialize return to position flags - winner will start returning in next frame
        winnerReturningToPosition = false;
        winnerReachedPosition = false;
        
        // Stop all movement - winner will start returning in animate() function
        winnerFighter.velocity.x = 0;
        winnerFighter.velocity.y = 0;
        loserFighter.velocity.x = 0;
        loserFighter.velocity.y = 0;
        
        console.log(`🏆 Winner (${winner}) will return to starting position and play victory animation`);
        
        determinewinner({ player, enemy });
    }
}

// Preload critical images (idle sprites) before starting animation
function preloadCriticalImages() {
    const criticalImages = [];
    
    // Collect player idle images (most important for initial display)
    if (player.sprites && player.sprites.idle && player.sprites.idle.images) {
        player.sprites.idle.images.forEach(img => {
            if (img) {
                criticalImages.push({img: img, type: 'player'});
            }
        });
    }
    
    // Collect enemy idle images (most important for initial display)
    if (enemy.sprites && enemy.sprites.idle && enemy.sprites.idle.images) {
        enemy.sprites.idle.images.forEach(img => {
            if (img) {
                criticalImages.push({img: img, type: 'enemy'});
            }
        });
    }
    
    console.log(`⏳ Preloading ${criticalImages.length} critical images (idle sprites)...`);
    
    if (criticalImages.length === 0) {
        console.log('✅ No critical images to preload, starting game immediately...');
        // Initialize round display
        if (typeof window.updateRoundDisplay === 'function') {
            window.updateRoundDisplay(1);
        }
        // Initialize finish move meters
        if (typeof window.updatePlayerFinishMoveMeter === 'function') {
            window.updatePlayerFinishMoveMeter(0);
        }
        if (typeof window.updateEnemyFinishMoveMeter === 'function') {
            window.updateEnemyFinishMoveMeter(0);
        }
        animate();
        return;
    }
    
    let loadedCount = 0;
    const totalImages = criticalImages.length;
    
    criticalImages.forEach((item, index) => {
        const img = item.img;
        if (img.complete && img.naturalWidth > 0) {
            loadedCount++;
            console.log(`✅ ${item.type} image ${index + 1}/${totalImages} already loaded`);
        } else {
            img.onload = function() {
                loadedCount++;
                console.log(`✅ ${item.type} image ${index + 1}/${totalImages} loaded (${this.naturalWidth}x${this.naturalHeight})`);
                if (loadedCount === totalImages) {
                    console.log('✅ All critical images preloaded! Starting game...');
                    // Ensure sprites are set
                    player.switchSprite('idle');
                    enemy.switchSprite('idle');
                    // Initialize round display
                    if (typeof window.updateRoundDisplay === 'function') {
                        window.updateRoundDisplay(1);
                    }
                    // Initialize finish move meters
                    if (typeof window.updatePlayerFinishMoveMeter === 'function') {
                        window.updatePlayerFinishMoveMeter(0);
                    }
                    if (typeof window.updateEnemyFinishMoveMeter === 'function') {
                        window.updateEnemyFinishMoveMeter(0);
                    }
                    animate();
                }
            };
            img.onerror = function() {
                loadedCount++;
                console.warn(`⚠️ ${item.type} image ${index + 1}/${totalImages} failed to load:`, this.src);
                if (loadedCount === totalImages) {
                    console.log('✅ All critical images processed! Starting game...');
                    player.switchSprite('idle');
                    enemy.switchSprite('idle');
                    // Initialize round display
                    if (typeof window.updateRoundDisplay === 'function') {
                        window.updateRoundDisplay(1);
                    }
                    // Initialize finish move meters
                    if (typeof window.updatePlayerFinishMoveMeter === 'function') {
                        window.updatePlayerFinishMoveMeter(0);
                    }
                    if (typeof window.updateEnemyFinishMoveMeter === 'function') {
                        window.updateEnemyFinishMoveMeter(0);
                    }
                    animate();
                }
            };
        }
    });
    
    // If all images are already loaded
    if (loadedCount === totalImages) {
        console.log('✅ All critical images already loaded! Starting game...');
        player.switchSprite('idle');
        enemy.switchSprite('idle');
        // Initialize round display
        if (typeof window.updateRoundDisplay === 'function') {
            window.updateRoundDisplay(1);
        }
        // Initialize finish move meters
        if (typeof window.updatePlayerFinishMoveMeter === 'function') {
            window.updatePlayerFinishMoveMeter(0);
        }
        if (typeof window.updateEnemyFinishMoveMeter === 'function') {
            window.updateEnemyFinishMoveMeter(0);
        }
        animate();
    }
}

// Start preloading critical images
preloadCriticalImages();

// Keyboard controls removed - game is now fully automatic
